#include "program_nrsample.h"
#include "program.h"
#include "propositional.h"
#include "fitness.h"
#include "bottom.h"

namespace lp {


	stat Program::generalize_id_nrsample(const sign_t& sign)
	{
		// Cache parameters
		const int param_csize = params.force_int(parameters::csize);
		const int param_noise = params.force_int(parameters::noise);
		const bool param_prune_visit = params.is_set(parameters::prune_visit);
		const bool param_prune_consistent = params.is_set(parameters::prune_consistent);
		const bool param_prune_inconsistent = params.is_set(parameters::prune_inconsistent);

		//std::cerr << "Generalize_nrsample\n";
		// Create constraints based on modes
		Vass vproto;
		ClauseDB allowed;
		int call_count = 0;
		PClause last_cl; // last inserted clause
		std::vector<Mode> id_modes; // modes for iterative deepening
		//std::vector<int> max_occurs_vec; 
		int current_depth = 0;
		// Get DPLL selection strategy
		unique_ptr<DPLL_Selection> select_ptr = get_dpll_selection(params);
		if (!select_ptr) {
			select_ptr.reset(new Simplicity_selection());
		}
		// Pick fitness function
		auto fitness = pick_fitness_function<bsf_type>(params);
		if (!fitness) {
			// Set default: lex
			fitness.reset(new fitness_lex<bsf_type>(params.force_int(parameters::terminate)));
		}

#ifndef NDEBUG
		vector<bitstring> history;
#endif

		auto init_search = [&]
		(Program& thy, Functor& bottom, int blen, const vector<Mode>& modes, 
			list<clause>& pex, list<Program::clause>& nex, Constraints& constr, stat& sts) -> bsf_type
		{
#ifndef NDEBUG
			history.clear();
#endif
			// Initialize ID-modes
			id_modes = modes;
			current_depth = 0;
			for (auto& m : id_modes) m.props[Mode::maxc] = 1;
			//max_occurs_vec.clear();
			//for (auto& m : modes) max_occurs_vec.push_back(m.props[Mode::maxc]);

			//std::cerr << "Calling init...\n";
			fitness->init(bottom,id_modes,thy,pex,nex,sts[stat::exc]);

			// Create prototype of model to store all settings
			vproto = Vass(bottom,id_modes,thy.params.force_int(parameters::csize),thy.params.force_int(parameters::dpllass));
			call_count = 0;
			last_cl.clear();
			allowed = ClauseDB(bottom,blen,id_modes,constr,thy.params);
			if (params.is_set(parameters::conditional_constraints)) {
				for (const auto& ip : thy.imp) {
					//std::cerr << "Adding implication constraints for " << ip.first.size() << " => " << ip.second << "\n";
					add_implication_constraints(make_literal_map(bottom),ip,allowed);
					//std::cerr << "Done\n";
				}
			}
			DEBUG_INFO(cerr << "Bottom Clause Constraints: " << allowed.size() << "\n");
			return bsf_type();
		};

		auto find_candidate = [&]
		(Program& thy, list<clause>& pex, list<Program::clause>& nex, Functor& bottom, int bsize, const vector<Mode>& modes, 
			const bsf_type& bsf, Constraints& constr, stat& sts, deadline_t deadline) -> bsf_type
		{
			// Get solution from DPLL
			Vass va;
			try {
				for (;;) {
					va = vproto;
					select_ptr->init(allowed,va,bsize);
					if (!allowed.simplify(va,bsize) || allowed.dpll(va,bsize,*select_ptr,deadline) <= 0) {
						// Increase max_occurs until it reaches actual max
						bool change = false;
						for (decltype(id_modes.size()) k = 1; k < id_modes.size(); ++k) {
							auto& limit = vproto.mode_limit(k-1);
							if (limit < id_modes[k].props[Mode::maxc]) {
								++limit;
								change = true;
							}
						}
						if (!change) {
							DEBUG_INFO(cout << "Exhausted search space\n");
							throw search_completed();
						} else {
							// Induce again
							continue;
						}
					}
					break;
				} // for
			} catch (max_assignments) {
				DEBUG_CRUCIAL(std::cerr << "Warning: maximal number of DPLL assignments reached\n");
				throw search_aborted();
			} catch (time_out) {
				DEBUG_CRUCIAL(std::cerr << "Warning: SAT solver timed out\n");
				throw search_aborted();
			}

			// DEBUGGING: check that we never revisit a point and respect csize
#ifndef NDEBUG
			const auto bs_tmp = make_bitstring(va);
			assert(find(history.begin(),history.end(),bs_tmp) == history.end());
			history.push_back(bs_tmp);
			assert( count(bs_tmp.begin(),bs_tmp.end(),1) <= param_csize );
#endif
			// Make candidate from bottom clause and mask
			bsf_type sol = make_bitstring(va);
			//std::cerr << "Masking bottom...\n";
			mask_bottom(bottom,sol.mask());
			assert( std::count_if(bottom.body_begin(),bottom.body_end(),[&](const Functor& t){ return !t.is_constant(true_id); }) 
				== std::count(sol.mask().begin(),sol.mask().end(),1) );

			++sts[stat::fitc]; // register fitness call
			//std::cerr << "Evaluating fitness for candidate: " << bottom << "\n";
			fitness->evaluate(bottom,sol,thy,pex,nex,bsf,sts[stat::exc],deadline);
			// Update consistent/inconsistent stats
			sol.is_consistent(param_noise) ? ++sts[stat::conc] : ++sts[stat::inconc];

			//DEBUG_INFO( cout << "Generated Solution: " << sol << "\n" );

			// Update Clause DB
			// Insert singleton constraint if requested
			if (param_prune_visit) {
				PClause cc = make_complement_clause(va,0);
				DEBUG_INFO( cout << "  Revisit Constraint: " << cc << "\n\n" );
				last_cl = cc;
				allowed.insert(move(cc));
			}
			if (sol.is_consistent(param_noise)) { 
				// Sol is Consistent
				if (param_prune_consistent) {
					PClause cc = make_complement_clause(va,1);
					DEBUG_INFO( cout << "  Prune down Constraint: " << cc << "\n\n" );
					last_cl = cc;
					allowed.insert(move(cc));
				}
			} else { 
				// Sol is Inconsistent
				if (param_prune_inconsistent) {
					PClause cc = make_complement_clause(va,-1);
					DEBUG_INFO( cout << "  Prune up Constraint: " << cc << "\n\n" );
					last_cl = cc;
					allowed.insert(move(cc));
				}
			}

			if (allowed.has_empty()) {
				DEBUG_INFO(cout << "Exhausted search space\n");
				throw sol; // last solution
			}

			return sol;
		};

		// Turn off clause learning for max_occurs as we iteratively construct
		decltype(this->generalize(sign,init_search,find_candidate)) res;
		const auto param_old_cll_max_occurs = params[parameters::dpll_cll_max_occurs];
		try {
			params[parameters::dpll_cll_max_occurs] = 0LL;
			res = this->generalize(sign,init_search,find_candidate);
			params[parameters::dpll_cll_max_occurs] = param_old_cll_max_occurs;
		} catch (...) {
			params[parameters::dpll_cll_max_occurs] = param_old_cll_max_occurs;
			throw;
		}
		return res;

	}


}



